<?php

namespace App\Controllers;

use App\Models\Client;
use App\Models\Meter;
use App\Models\Payment;

class FrontController extends Controller
{
    function index()
    {
        return $this->view('front/index', ['title' => 'Inicio'], 'front');
    }

    function consulta()
    {
        return $this->view('consulta/consulta', ['title' => 'Inicio'], 'front');
    }


    public function consultaFactura()
    {
        // 1. Validar y sanitizar inputs
        $medidor = trim($_REQUEST["medidor"] ?? '');
        $documento = trim($_REQUEST["documento"] ?? '');

        // Validar que al menos uno de los dos campos esté presente
        if (empty($medidor) && empty($documento)) {
            return $this->jsonResponse('error', 'Debe proporcionar al menos número de medidor o documento de identidad');
        }

        // 2. Caso 1: Solo medidor (no necesita documento)
        if (!empty($medidor) && empty($documento)) {
            $meterData = (new Meter())->mostrarPorSerie($medidor);
            if (!$meterData) {
                return $this->jsonResponse('error', 'No se encontraron registros');
            }

            // Obtener datos del cliente asociado al medidor
            $clientData = (new Client())->mostrar($meterData['client_id']);
            if (!$clientData) {
                return $this->jsonResponse('error', 'No se encontraron registros');
            }

            return $this->procesarRespuesta($clientData, $meterData);
        }

        // 3. Caso 2: Solo documento o ambos
        $clientData = (new Client())->mostrarPorDocumento($documento);
        if (!$clientData) {
            return $this->jsonResponse('error', 'No se encontraron registros');
        }

        // Si se envió solo documento, devolver lista de medidores
        if (empty($medidor)) {
            $medidoresCliente = (new Meter())->mostrarPorCliente($clientData['id']);

            if (empty($medidoresCliente)) {
                return $this->jsonResponse('error', 'No se encontraron registros');
            }

            $medidores = [];
            foreach ($medidoresCliente as $medidor) {
                $medidores[] = [
                    'serie' => $medidor['series'],
                    'ubicacion' => $medidor['location']
                ];
            }

            return $this->jsonResponse('success', '', ['medidores' => $medidores]);
        }

        // Si se envió ambos, verificar que el medidor pertenece al cliente
        $meterData = (new Meter())->mostrarPorSerie($medidor);
        if (!$meterData) {
            return $this->jsonResponse('error', 'No se encontraron registros');
        }

        if ($meterData['client_id'] != $clientData['id']) {
            return $this->jsonResponse('error', 'No se encontraron registros');
        }

        return $this->procesarRespuesta($clientData, $meterData);
    }

    // Función auxiliar para procesar la respuesta final
    private function procesarRespuesta($clientData, $meterData)
    {
        $payment = new Payment();
        $latestInvoice = $payment->mostrarUltimaFactura('meter_readings', $meterData['id']);

        $facturaPendiente = $payment->totalPagoPendienteAgua('meter_readings', $clientData['id'], $meterData['id']) ?? 0;
        $multaPendiente = $payment->totalPagoPendienteMulta('client_meeting', $clientData['id']) ?? 0;

        if (!$latestInvoice) {
            return $this->jsonResponse('info', 'No se encontraron registros');
        }

        $meterReading = $payment->mostrarLecturaMedidor($latestInvoice['reference_id']);
        if (!$meterReading) {
            return $this->jsonResponse('error', 'No se encontraron registros');
        }

        $response = [
            'status' => 'success',
            'cliente' => [
                'nombre' => $clientData['name'] ?? '',
                'documento' => $clientData['document'] ?? '',
                'direccion' => $clientData['address'] ?? ''
            ],
            'medidor' => [
                'serie' => $meterData['series'] ?? '',
                'ubicacion' => $meterData['location'] ?? ''
            ],
            'ultima_factura' => [
                'uuid' => $latestInvoice['uuid'] ?? null,
                'fecha' => $latestInvoice['fecha_pago'] ?? '',
                'periodo' => $meterReading['fecha_pago'] ?? '',
                'consumo' => $meterReading['consumption'] ?? 0,
                'monto' => $latestInvoice['monto'] ?? 0,
                'estado' => $this->getEstadoPago($latestInvoice['estado'] ?? 0)
            ],
            'estado_cuenta' => [
                'monto_pendiente' =>  $facturaPendiente + $multaPendiente,
                'moneda' => 'USD'
            ],
            'historial_facturas' => $payment->listarFacturas($clientData['id'], $meterData['id']) ?? [],
            'historial_multas' => $payment->listarFacturaReuniones($clientData['id']) ?? []
        ];

        return $this->jsonResponse('success', '', $response);
    }

    // Método auxiliar para respuestas JSON
    private function jsonResponse($status, $message = '', $data = [])
    {
        header('Content-Type: application/json');
        echo json_encode([
            'status' => $status,
            'message' => $message,
            'data' => $data
        ]);
        exit;
    }

    // Método auxiliar para estado de pago
    private function getEstadoPago($status)
    {
        $estados = [
            0 => 'Pendiente',
            1 => 'Pagado',
            2 => 'Vencido',
            3 => 'Cancelado'
        ];
        return $estados[$status] ?? 'Desconocido';
    }
}
